<?PHP
	/**
	* de_error_handler
	*
	* @package Error Handler
	* @category Kernel
	* @author Denker Interaktiv GmbH
	* @version 1.1
	* @link https://denker-interaktiv.de
	*/

	# Weiterhin einbauen:
	# 	1. Weiterleitung auf fehler Seite
	# 	2. Logging an aus
	# 	3. E-Mailversand
	# 	4. Fehler ausgeben im Template
	#	5. Auflogin hören
	#  6. 404 Forwarding etc.

	define('TYPEHINT_PCRE','/^Argument (\d)+ passed to (?:(\w+)::)?(\w+)\(\) must be an instance of (\w+), (\w+) given/');
	error_reporting(E_ALL);
 	ini_set('display_errors', 1);

	class de_error_handler
	{
		/*
		* Behandelt alle Arten von Fehlern, sowohl von PHP als auch vom User erzeugt.
		* Funktionen: Speichern in DB, Log in txt, Print, Weiterleitung im Production mode, E-Mail, Error 404 u.A. Weiterleitung
		*/
		private $debug = 0;
		private $user_id;
		private $mode;
		private $error_alerts=array();
		public 	$exception_alerts = array();
		public  $var_dump_alerts = array();
		public  $print_r_alerts = array();
		public $rendertime = 0;

		# Nur die letzten $max_logentires in der Datenbank speichern, die alten Löschen
		# 0 = Keine Einträge werden gelöscht
		private $max_logentries = 40;

		private $db_host;
		private $db_db;
		private $db_user;
		private $db_password;

		#Wie sollen Fehler ausgegeben werden: Log, DB, Mail, HTML etc.
		private $output_type;


		private static $type_hints = array
		(
			'boolean'   => 'is_bool',
			'integer'   => 'is_int',
			'float'     => 'is_float',
			'string'    => 'is_string',
			'resrouce'  => 'is_resource'
		);

		public function __construct($mysql, $output_type = NULL)
		{
			#$this->debug = $debug;
			$this->rendertime = microtime(true);
			$this->db_host = $mysql["HOST"];
			$this->db_db = $mysql["DATABASE"];
			$this->db_user = $mysql["USER"];
			$this->db_password = $mysql["PASSWORD"];

			if($output_type != NULL)
			{
				$this->output_type = $output_type;
			}

			//URL aufgeteilt
			if(isset($_GET["url"]))
			{
				$exploded_url = explode("/",$_GET["url"]);


			}
			set_error_handler(array($this, "handle_error"));
			set_exception_handler(array($this, "handle_exception"));
			register_shutdown_function(array($this, "handle_shutdown"));
		}

		public static function error ($error_string,$error_class,$error_function, $error_type)
		{
			trigger_error(
				json_encode(
					array
					(
						"ERROR_STRING" => $error_string,
						"ERROR_CLASS" => $error_class,
						"ERROR_FUNCTION" => $error_function
					)
				),$error_type
			);
		}


		public function handle_error ($error_type, $error_string, $error_file_string, $error_line)
		{
			//echo $error_type."<br>";
			$type_hinting=false;
			$error_file_explode=explode("/",$error_file_string);
			$error_file = str_replace(SYSTEM_PATH,"",$error_file_string);
			$error_request = json_encode($_REQUEST);
			static $db;
			if (empty($db))
			{

				$db = new PDO("mysql:host={$this->db_host};dbname={$this->db_db}", $this->db_user, $this->db_password);
			}

			$query = "INSERT INTO ".DB_PREFIX."Log (timestamp,type,request,file,class,function,line,message,user_id) VALUES (NOW(),? ,? ,? ,? ,? ,? ,? ,1)";
			$stmt = $db->prepare($query);

			switch ($error_type)
			{
				case 2:
					$error_written_type = "E_WARNING";
					$error_color="orange";
					$error_icon="warning";
				break;

				case 8:
					$error_written_type = "E_NOTICE";
					$error_color="pink";
					$error_icon="info";
				break;

				case 256:
					$error_written_type = "E_USER_ERROR";
					$error_string = json_decode($error_string);
					$error_class = $error_string->ERROR_CLASS;
					$error_function = $error_string->ERROR_FUNCTION;
					$error_string = $error_string->ERROR_STRING;
					$error_color="red";
					$error_icon="fire";
				break;

				case 512:
					$error_written_type = "E_USER_WARNING";
					$error_string = json_decode($error_string);
					$error_class = $error_string->ERROR_CLASS;
					$error_function = $error_string->ERROR_FUNCTION;
					$error_string = $error_string->ERROR_STRING;
					$error_color="orange";
					$error_icon="warning";
				break;

				case 1024:
					$error_written_type = "E_USER_NOTICE";
					$error_string = json_decode($error_string);
					$error_class = $error_string->ERROR_CLASS;
					$error_function = $error_string->ERROR_FUNCTION;
					$error_string = $error_string->ERROR_STRING;
					$error_color="pink";
					$error_icon="info";
				break;

				case 4096:
					$error_written_type = "E_RECOVERABLE_ERROR";
					$error_color="red";
					$error_icon="fire";
				break;

				case 8191:
					$error_written_type = "E_ALL";
					$error_color="red";
					$error_icon="fire";
				break;

				case "555":
					$error_string = json_decode($error_string);
					$error_class = $error_string->ERROR_CLASS;
					$error_function = $error_string->ERROR_FUNCTION;
					$error_string = $error_string->ERROR_STRING;
				break;

			}
			if( ! isset($error_class)) $error_class = "";
			if( ! isset($error_function)) $error_function = "";
			switch ($error_type)
			{
				case E_NOTICE:
				case E_USER_NOTICE:
				case E_STRICT:
					$stmt->execute(array("NOTICE", $error_request, $error_file, $error_class, $error_function, $error_line, $error_string));
				break;

				case E_WARNING:
				case E_USER_WARNING:
					$stmt->execute(array("WARNING", $error_request, $error_file, $error_class, $error_function, $error_line, $error_string));
				break;

				case E_ERROR:
				case E_USER_ERROR:
					$stmt->execute(array("FATAL", $error_request, $error_file, $error_class, $error_function, $error_line, $error_string));

				break;

				case E_RECOVERABLE_ERROR:
					if (preg_match(TYPEHINT_PCRE, $error_string, $error_matches))
					{
						list($error_matched_string, $error_arg_index, $error_matched_class, $error_matched_function, $error_hint, $error_matched_type) = $error_matches;

						if (isset(self::$type_hints[$error_hint]))
						{

							$error_backtrace = debug_backtrace();
							$error_passed_variable  = NULL;

							if (self::get_typehinted_argument($error_backtrace, $error_matched_class,$error_matched_function, $error_arg_index, $error_passed_variable))
							{
								if (call_user_func(self::$type_hints[$error_hint], $error_passed_variable))
								{
									$type_hinting = true;
								}
								else
								{
									$type_hinting = false;
									$error_string = "Argument $error_arg_index passed to $error_matched_class::$error_matched_function must be $error_hint, $error_matched_type given.";
									$stmt->execute(array("RECOVERABLE_ERROR", $error_request, $error_file, $error_class, $error_function, $error_line, $error_string));

								}
							}
						}
					}

				break;

				case 555:
					$type_hinting = true;
					$stmt->execute(array("EXCEPTION", $error_request, $error_file, $error_class, $error_function, $error_line, $error_string));
				break;

				default:
					exit("Unknown error at $error_file:$error_line");
				break;


			}

			if($type_hinting == false)
			{
				array_push($this->error_alerts,array
				(
				"error_color" => $error_color,
				"error_icon" => $error_icon,
				"error_written_type" => $error_written_type,
				"error_type" => $error_type,
				"error_line" => $error_line,
				"error_file" => $error_file,
				"error_class" => $error_class,
				"error_function" => $error_function,
				#$_SERVER["DOCUMENT_ROOT"] Einbinden
				"error_string" => str_replace("href='", "href='http://php.net/", $error_string)
				)
				);
			}
		}

		public function handle_exception($exception)
		{
			array_push($this->exception_alerts, array
			(
				"variable_name" => "EXCEPTION",
				"variable_dump" => strip_tags(grab_print_r($exception))
			));

			$trace_info = $exception->getTrace();

			$message = json_encode(
					array
					(
						"ERROR_STRING" => $exception->getMessage(),
						"ERROR_CLASS" => $trace_info[0]["class"],
						"ERROR_FUNCTION" => $trace_info[0]["function"]
					)
			);
			$this->handle_error(555,$message,$exception->getFile(),$exception->getLine());

		}

		# UNCATCHABLE ERRORS
		public static function handle_shutdown()
		{
			$error = error_get_last();
			if($error)
			{
				//URL aufgeteilt
				if(isset($_GET["url"]))
				{
					$exploded_url = explode("/",$_GET["url"]);


				}

				# IF YOU WANT TO CLEAR ALL BUFFER, UNCOMMENT NEXT LINE:
				//ob_end_clean();
				switch ($error['type'])
				{
					case E_ERROR:
						$error_type = "E_ERROR";
					break;

					case E_CORE_ERROR:
						$error_type = "E_CORE_ERROR";
					break;

					case E_COMPILE_ERROR:
						$error_type = "E_COMPILE_ERROR";
					break;

					case E_USER_ERROR:
						$error_type = "E_USER_ERROR";
					break;

					case E_RECOVERABLE_ERROR:
						$error_type = "E_RECOVERABLE_ERROR";
					break;

					case E_CORE_WARNING:
						$error_type = "E_CORE_WARNING";
					break;

					case E_COMPILE_WARNING:
						$error_type = "E_COMPILE_WARNING";
					break;

					case E_PARSE:
						$error_type = "E_PARSE";
					break;

					case E_NOTICE:
						$error_type = "E_NOTICE";
					break;

					case E_USER_NOTICE:
						$error_type = "E_USER_NOTICE";
					break;

					default:
						$error_type = "UNKNOWN";
					break;
				}

				 $shutdown_alert = array
				(
				"variable_name" => "SHUTDOWN ".$error_type,
				"variable_dump" => strip_tags(grab_print_r($error))
				);

				$error_alerts=$GLOBALS["error_handler"]->error_alerts;
				$exception_alerts=$GLOBALS["error_handler"]->exception_alerts;
				$var_dump_alerts=$GLOBALS["error_handler"]->var_dump_alerts;
				$print_r_alerts=$GLOBALS["error_handler"]->print_r_alerts;

				// Load Composer
				include(dirname(__FILE__).'/../vendor/autoload.php');

				// Load .env
				$dotenv = Dotenv\Dotenv::createImmutable(dirname(__FILE__).'/../../');
				$dotenv->load(true);

				$db = new PDO("mysql:host=".$_ENV['DB_HOST'].";dbname=".$_ENV['DB_DATABASE'], $_ENV['DB_USERNAME'],$_ENV['DB_PASSWORD']);

				$query = "INSERT INTO ".DB_PREFIX."Log (timestamp,type,request,file,class,function,line,message,user_id) VALUES (NOW(),? ,? ,? ,? ,? ,? ,? ,1)";
				$stmt = $db->prepare($query);
				$stmt->execute(array("SHUTDOWN ".$error_type,json_encode($_REQUEST),$error["file"],"","",$error["line"], $error["message"]));
				$rendertime = $GLOBALS["error_handler"]->get_rendertime();


			}
			else
			{
				return true;
			}
		}

		private static function get_typehinted_argument($error_backtrace, $error_class, $error_function, $error_arg_index, &$error_passed_variable)
		{

			foreach ($error_backtrace as $error_trace)
			{
				# Match the function; Note we could do more defensive error checking.
				if (isset($error_trace['function']) && $error_trace['function'] == $error_function)
				{
					$error_passed_variable = $error_trace['args'][$error_arg_index - 1];
					return TRUE;
				}
			}

			return FALSE;
		}

		public function __destruct()
		{
			if ( ! headers_sent()) {
				#header("Content-Type: text/html; charset=utf-8");
			}

			if( count($this->exception_alerts))
			{
				#ob_end_clean();
				$error_alerts=$this->error_alerts;
				$exception_alerts=$this->exception_alerts;
				$var_dump_alerts=$this->var_dump_alerts;
				$print_r_alerts=$this->print_r_alerts;
			}


		}

		public function clear_log_table()
		{

		}

		public function get_error_handler_content()
		{
			if ( ! headers_sent()) {
				header("Content-Type: text/html; charset=utf-8");
			}
			if(!isset($_GET["CRON"]))
			{
				if(count($this->error_alerts) !=0 || count($this->exception_alerts)  !=0 || count($this->var_dump_alerts)  !=0 || count($this->print_r_alerts)  !=0)
				{
					#ob_end_clean();
					$error_alerts=$this->error_alerts;
					$exception_alerts=$this->exception_alerts;
					$var_dump_alerts=$this->var_dump_alerts;
					$print_r_alerts=$this->print_r_alerts;

					ob_start();
					$output = ob_get_clean();

					return $output;
				}
			}

		}

		private function get_rendertime()
		{
			$end = microtime(true);
			$creationtime = ($end - $this->rendertime);
			return "Seite wurde in {$creationtime} Sekunden generiert.";
		}

		private function check_https ()
		{
			if (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443)
			{
				return TRUE;
			}
			else
			{
				return FALSE;
			}
		}
	}

?>
